const API_BASE = 'https://sqzd.in';

// DOM elements
const loading = document.getElementById('loading');
const loginPrompt = document.getElementById('login-prompt');
const formContainer = document.getElementById('form-container');
const form = document.getElementById('shorten-form');
const longUrlInput = document.getElementById('long-url');
const customCodeInput = document.getElementById('custom-code');
const submitBtn = document.getElementById('submit-btn');
const result = document.getElementById('result');
const shortUrlInput = document.getElementById('short-url');
const copyBtn = document.getElementById('copy-btn');
const copyFeedback = document.getElementById('copy-feedback');
const newLinkBtn = document.getElementById('new-link-btn');
const error = document.getElementById('error');
const errorMessage = document.getElementById('error-message');
const retryBtn = document.getElementById('retry-btn');

// Check if user is logged in by trying to fetch their URLs
async function checkAuth() {
  try {
    const response = await fetch(`${API_BASE}/api/urls?limit=1`, {
      credentials: 'include'
    });
    return response.ok;
  } catch {
    return false;
  }
}

// Get current tab URL
async function getCurrentTabUrl() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  return tab?.url || '';
}

// Show/hide sections
function showSection(section) {
  loading.style.display = 'none';
  loginPrompt.style.display = 'none';
  formContainer.style.display = 'none';

  if (section === 'loading') {
    loading.style.display = 'block';
  } else if (section === 'login') {
    loginPrompt.style.display = 'block';
  } else if (section === 'form') {
    formContainer.style.display = 'block';
    form.style.display = 'block';
    result.style.display = 'none';
    error.style.display = 'none';
  } else if (section === 'result') {
    formContainer.style.display = 'block';
    form.style.display = 'none';
    result.style.display = 'block';
    error.style.display = 'none';
  } else if (section === 'error') {
    formContainer.style.display = 'block';
    form.style.display = 'none';
    result.style.display = 'none';
    error.style.display = 'block';
  }
}

// Create short URL
async function createShortUrl(longUrl, customCode) {
  const response = await fetch(`${API_BASE}/api/urls`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    credentials: 'include',
    body: JSON.stringify({
      longUrl,
      customCode: customCode || undefined,
    }),
  });

  const data = await response.json();

  if (!response.ok) {
    throw new Error(data.error || 'Failed to create short URL');
  }

  return data.url;
}

// Handle form submission
form.addEventListener('submit', async (e) => {
  e.preventDefault();

  const longUrl = longUrlInput.value;
  const customCode = customCodeInput.value.trim();

  submitBtn.disabled = true;
  submitBtn.textContent = 'Creating...';

  try {
    const url = await createShortUrl(longUrl, customCode);
    shortUrlInput.value = `${API_BASE}/${url.short_code}`;
    copyFeedback.textContent = '';
    showSection('result');
  } catch (err) {
    errorMessage.textContent = err.message;
    showSection('error');
  } finally {
    submitBtn.disabled = false;
    submitBtn.textContent = 'Shorten URL';
  }
});

// Copy to clipboard
copyBtn.addEventListener('click', async () => {
  try {
    await navigator.clipboard.writeText(shortUrlInput.value);
    copyFeedback.textContent = 'Copied to clipboard!';
    setTimeout(() => {
      copyFeedback.textContent = '';
    }, 2000);
  } catch {
    copyFeedback.textContent = 'Failed to copy';
  }
});

// New link button
newLinkBtn.addEventListener('click', () => {
  customCodeInput.value = '';
  showSection('form');
});

// Retry button
retryBtn.addEventListener('click', () => {
  showSection('form');
});

// Initialize
async function init() {
  showSection('loading');

  const isLoggedIn = await checkAuth();

  if (!isLoggedIn) {
    showSection('login');
    return;
  }

  const url = await getCurrentTabUrl();
  longUrlInput.value = url;
  showSection('form');
}

init();
